<?php
/**
 * Form Handler with PHPMailer SMTP (matches send.php setup)
 * Handles contact, waitlist, and adoption application forms
 */

ob_start();

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

ini_set('display_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Load configuration from site-config.json
$configFile = __DIR__ . '/../config/site-config.json';
$config = [];

if (file_exists($configFile)) {
    $configJson = file_get_contents($configFile);
    $config = json_decode($configJson, true);
}

// Get config values
$adminEmail = isset($config['email']) ? $config['email'] : 'noreply@example.com';
$siteName = isset($config['title']) ? $config['title'] : 'Cuddly Dachshunds';
$siteTagline = isset($config['tagline']) ? $config['tagline'] : '';
$sitePhone = isset($config['phone']) ? $config['phone'] : '';
$fromEmail = isset($config['fromEmail']) ? $config['fromEmail'] : 'info@test.123autodealsllc.com';
// Small delay (milliseconds) before sending confirmation emails to users
// Set to 0 to disable. Default: 500ms
$confirmationDelayMs = isset($config['confirmationDelayMs']) ? intval($config['confirmationDelayMs']) : 500;

// Load SMTP config (separate file) - fall back to sensible defaults
$smtpConfigFile = __DIR__ . '/../config/smtp-config.json';
$smtpConfig = [];
if (file_exists($smtpConfigFile)) {
    $smtpJson = file_get_contents($smtpConfigFile);
    $smtpConfig = json_decode($smtpJson, true) ?: [];
}

$smtpHost = isset($smtpConfig['host']) ? $smtpConfig['host'] : 'smtp.hostinger.com';
$smtpUser = isset($smtpConfig['username']) ? $smtpConfig['username'] : 'info@test.123autodealsllc.com';
$smtpPass = isset($smtpConfig['password']) ? $smtpConfig['password'] : 'itsjustme-C300';
$smtpEncryption = isset($smtpConfig['encryption']) ? strtolower($smtpConfig['encryption']) : 'smtps';
$smtpPort = isset($smtpConfig['port']) ? intval($smtpConfig['port']) : 465;
$smtpTimeout = isset($smtpConfig['timeout']) ? intval($smtpConfig['timeout']) : 10;

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    ob_end_clean();
    exit(json_encode(['status' => 'ok']));
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    ob_end_clean();
    exit(json_encode(['success' => false, 'message' => 'Method not allowed']));
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if ($input === null) {
    http_response_code(400);
    ob_end_clean();
    exit(json_encode(['success' => false, 'message' => 'Invalid JSON input']));
}

// Determine form type
$formType = isset($input['formType']) ? $input['formType'] : '';

// Helper functions
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

function sanitizeInput($input) {
    return htmlspecialchars(stripslashes(trim($input)), ENT_QUOTES, 'UTF-8');
}

function getClientGeolocation() {
    $ip = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    
    // Handle proxies and load balancers
    if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
        $ip = $_SERVER['HTTP_CF_CONNECTING_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        $ip = trim($ips[0]);
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_FORWARDED_FOR'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED'])) {
        $ip = $_SERVER['HTTP_FORWARDED'];
    }
    
    // Default location for localhost
    if ($ip === '127.0.0.1' || $ip === 'localhost') {
        return [
            'country' => 'Local',
            'city' => 'Localhost',
            'region' => 'N/A',
            'ip' => $ip
        ];
    }
    
    // Call free IP geolocation API
    $cacheFile = __DIR__ . '/../logs/geo_cache_' . md5($ip) . '.json';
    
    // Check cache first (valid for 7 days)
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < 604800) {
        $cached = json_decode(file_get_contents($cacheFile), true);
        if ($cached) return $cached;
    }
    
    try {
        // Try ip-api.com (free tier, no API key required)
        $url = "http://ip-api.com/json/$ip?fields=status,country,city,regionName,query";
        $context = stream_context_create([
            'http' => [
                'timeout' => 3,
                'user_agent' => 'Cuddly Dachshunds Contact Form'
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response) {
            $data = json_decode($response, true);
            
            if ($data && $data['status'] === 'success') {
                $result = [
                    'country' => $data['country'] ?? 'Unknown',
                    'city' => $data['city'] ?? 'Unknown',
                    'region' => $data['regionName'] ?? 'Unknown',
                    'ip' => $data['query'] ?? $ip
                ];
                
                // Cache the result
                @mkdir(__DIR__ . '/../logs', 0755, true);
                @file_put_contents($cacheFile, json_encode($result));
                
                return $result;
            }
        }
    } catch (Exception $e) {
        error_log('Geolocation Error: ' . $e->getMessage());
    }
    
    // Fallback if API fails
    return [
        'country' => 'Unknown',
        'city' => 'Unknown',
        'region' => 'Unknown',
        'ip' => $ip
    ];
}

function sendPHPMailerEmail($mail, $to, $subject, $htmlBody, $replyToEmail = '', $replyToName = '', $fromName = '') {
    try {
        // Configure SMTP using config file values (fallbacks above)
        global $smtpHost, $smtpUser, $smtpPass, $smtpEncryption, $smtpPort, $smtpTimeout, $fromEmail;
        $mail->isSMTP();
        $mail->Host       = $smtpHost;
        $mail->SMTPAuth   = true;
        $mail->Username   = $smtpUser;
        $mail->Password   = $smtpPass;
        // Map encryption string to PHPMailer constants
        if ($smtpEncryption === 'smtps' || $smtpEncryption === 'ssl') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        } else if ($smtpEncryption === 'starttls' || $smtpEncryption === 'tls') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        }
        $mail->Port       = $smtpPort;
        $mail->Timeout    = $smtpTimeout;

        // Clear any previous recipients
        $mail->clearAddresses();
        $mail->clearReplyTos();

        // Set From to the authenticated SMTP username to ensure the From address is authenticated
        global $smtpUser;
        $fromDisplayName = !empty($fromName) ? $fromName : 'Cuddly Dachshunds';
        $authFrom = !empty($smtpUser) ? $smtpUser : $fromEmail;
        $mail->setFrom($authFrom, $fromDisplayName);
        $mail->addAddress($to);

        // Set Reply-To if provided
        if (!empty($replyToEmail) && isValidEmail($replyToEmail)) {
            $mail->addReplyTo($replyToEmail, $replyToName);
        }

        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $htmlBody;

        // Send and return status
        return $mail->send();
    } catch (Exception $e) {
        error_log('PHPMailer Error: ' . $e->getMessage());
        return false;
    }
}

try {
    // Check PHPMailer Paths
    if (!file_exists(__DIR__ . '/../PHPMailer/src/PHPMailer.php')) {
        throw new Exception("PHPMailer files not found. Check that the 'PHPMailer/src' directory exists.");
    }
    require __DIR__ . '/../PHPMailer/src/Exception.php';
    require __DIR__ . '/../PHPMailer/src/PHPMailer.php';
    require __DIR__ . '/../PHPMailer/src/SMTP.php';

    $mail = new PHPMailer(true);

    // ===== HANDLE CONTACT FORM =====
    if ($formType === 'contact') {
        $required_fields = ['firstName', 'lastName', 'email', 'phone', 'subject', 'message'];
        $errors = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || empty(trim($input[$field]))) {
                $errors[] = ucfirst($field) . ' is required';
            }
        }

        if (isset($input['email']) && !isValidEmail($input['email'])) {
            $errors[] = 'Invalid email address';
        }

        if (!empty($errors)) {
            http_response_code(400);
            ob_end_clean();
            exit(json_encode(['success' => false, 'message' => 'Validation failed', 'errors' => $errors]));
        }

        $firstName = sanitizeInput($input['firstName']);
        $lastName = sanitizeInput($input['lastName']);
        $email = sanitizeInput($input['email']);
        $phone = sanitizeInput($input['phone']);
        $subject = sanitizeInput($input['subject']);
        $message = sanitizeInput($input['message']);

        // Get geolocation
        $geo = getClientGeolocation();

        // Send to admin
        $adminSubject = "New Contact Form Submission - $subject";
        $adminBody = "
            <html><body>
            <h2>$siteName - New Contact Form Submission</h2>
            <hr>
            <p><strong>Name:</strong> $firstName $lastName</p>
            <p><strong>Email:</strong> $email</p>
            <p><strong>Phone:</strong> $phone</p>
            <p><strong>Subject:</strong> $subject</p>
            <p><strong>Message:</strong></p>
            <p>" . nl2br($message) . "</p>
            <p><strong>Submitted:</strong> " . date('Y-m-d H:i:s') . "</p>
            <p><strong>Sender Location:</strong> {$geo['city']}, {$geo['region']}, {$geo['country']}</p>
            <p><strong>Sender IP:</strong> {$geo['ip']}</p>
            <hr>
            </body></html>
        ";

        $adminSent = sendPHPMailerEmail($mail, $adminEmail, $adminSubject, $adminBody, $email, "$firstName $lastName", "$firstName $lastName");

        if (!$adminSent) {
            throw new Exception('Failed to send admin notification');
        }

        // Send confirmation to user
        $confirmSubject = "We Received Your Message - $siteName";
        $confirmBody = "
            <html><body>
            <h2>Thank You, $firstName!</h2>
            <p>We've received your message and will get back to you within 24 hours.</p>
            <p><strong>Your Message Details:</strong></p>
            <p><strong>Subject:</strong> $subject</p>
            <p><strong>Message:</strong></p>
            <p>" . nl2br($message) . "</p>
            <p>If you have any questions in the meantime, feel free to reach out at $sitePhone or $fromEmail.</p>
            <hr>
            <p><small>$siteName — $siteTagline</small></p>
            <p><small>This is an automated confirmation email. Please do not reply to this message.</small></p>
            </body></html>
        ";

        // Optional small delay before sending confirmation to user
        if (!empty($confirmationDelayMs) && is_numeric($confirmationDelayMs) && $confirmationDelayMs > 0) {
            usleep(intval($confirmationDelayMs) * 1000); // usleep expects microseconds
        }
        // Optional small delay before sending reservation confirmation to applicant
        if (!empty($confirmationDelayMs) && is_numeric($confirmationDelayMs) && $confirmationDelayMs > 0) {
            usleep(intval($confirmationDelayMs) * 1000);
        }
        sendPHPMailerEmail($mail, $email, $confirmSubject, $confirmBody);
        logSubmission('contact', $firstName, $lastName, $email, $phone, $subject);

        http_response_code(200);
        ob_end_clean();
        exit(json_encode(['success' => true, 'message' => 'Your message has been sent successfully!']));
    }

    // ===== HANDLE VIP WAITLIST FORM =====
    else if ($formType === 'waitlist') {
        if (!isset($input['email']) || empty(trim($input['email']))) {
            http_response_code(400);
            ob_end_clean();
            exit(json_encode(['success' => false, 'message' => 'Email is required']));
        }

        if (!isValidEmail($input['email'])) {
            http_response_code(400);
            ob_end_clean();
            exit(json_encode(['success' => false, 'message' => 'Invalid email address']));
        }

        $email = sanitizeInput($input['email']);

        // Get geolocation
        $geo = getClientGeolocation();

        // Send to admin
        $adminSubject = "New VIP Waitlist Subscription";
        $adminBody = "
            <html><body>
            <h2>$siteName - New VIP Waitlist Subscription</h2>
            <hr>
            <p><strong>Email:</strong> $email</p>
            <p><strong>Subscribed:</strong> " . date('Y-m-d H:i:s') . "</p>
            <p><strong>Subscriber Location:</strong> {$geo['city']}, {$geo['region']}, {$geo['country']}</p>
            <p><strong>Subscriber IP:</strong> {$geo['ip']}</p>
            <hr>
            </body></html>
        ";

        $adminSent = sendPHPMailerEmail($mail, $adminEmail, $adminSubject, $adminBody, $email, '');

        if (!$adminSent) {
            throw new Exception('Failed to send admin notification');
        }

        // Send welcome email to subscriber
        $welcomeSubject = "Welcome to $siteName VIP Waitlist!";
        $welcomeBody = "
            <html><body>
            <h2>Welcome to the $siteName VIP Waitlist!</h2>
            <p><em>$siteTagline</em></p>
            <p>Thank you for joining the $siteName VIP waitlist.</p>
            <p>You'll be the first to know when new litters arrive and about exclusive adoption events!</p>
            <p><strong>What's Next?</strong></p>
            <ul>
                <li>Watch your inbox for litter announcements</li>
                <li>Receive early access to available puppies</li>
                <li>Get special VIP offers and updates</li>
            </ul>
            <p>If you have any questions, feel free to reach out at $sitePhone or $fromEmail.</p>
            <hr>
            <p><small>$siteName — $siteTagline</small></p>
            <p><small>This is an automated confirmation email. Please do not reply to this message.</small></p>
            </body></html>
        ";

        // Optional small delay before sending welcome email to subscriber
        if (!empty($confirmationDelayMs) && is_numeric($confirmationDelayMs) && $confirmationDelayMs > 0) {
            usleep(intval($confirmationDelayMs) * 1000);
        }
        sendPHPMailerEmail($mail, $email, $welcomeSubject, $welcomeBody);
        logSubmission('waitlist', '', '', $email, '', '');

        http_response_code(200);
        ob_end_clean();
        exit(json_encode(['success' => true, 'message' => 'You\'ve been added to our VIP waitlist!']));
    }

    // ===== HANDLE ADOPTION APPLICATION FORM =====
    else if ($formType === 'reservation') {
        $required_fields = ['full_name', 'email', 'phone', 'address'];
        $errors = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || empty(trim($input[$field]))) {
                $errors[] = ucfirst(str_replace('_', ' ', $field)) . ' is required';
            }
        }

        if (isset($input['email']) && !isValidEmail($input['email'])) {
            $errors[] = 'Invalid email address';
        }

        if (!empty($errors)) {
            http_response_code(400);
            ob_end_clean();
            exit(json_encode(['success' => false, 'message' => 'Validation failed', 'errors' => $errors]));
        }

        // Sanitize inputs
        $fullName = sanitizeInput($input['full_name']);
        $email = sanitizeInput($input['email']);
        $phone = sanitizeInput($input['phone']);
        $address = sanitizeInput($input['address']);
        $message = isset($input['message']) ? sanitizeInput($input['message']) : '';
        $puppyName = isset($input['puppyName']) ? sanitizeInput($input['puppyName']) : 'A Puppy';
        $puppySlug = isset($input['puppySlug']) ? sanitizeInput($input['puppySlug']) : '';
        $deliveryMethod = isset($input['delivery_method']) ? sanitizeInput($input['delivery_method']) : 'pickup';
        $shippingAddress = isset($input['shipping_address']) ? sanitizeInput($input['shipping_address']) : '';
        $depositMethod = isset($input['deposit_method']) ? sanitizeInput($input['deposit_method']) : '';
        $shippingFee = isset($input['shipping_fee']) ? floatval($input['shipping_fee']) : 0;

        // Get geolocation
        $geo = getClientGeolocation();

        // Send to admin
        $adminSubject = "New Puppy Reservation - $puppyName";
        $adminBody = "
            <html><body>
            <h2>$siteName - New Puppy Reservation</h2>
            <hr>
            <p><strong>Puppy:</strong> $puppyName ($puppySlug)</p>
            <p><strong>Applicant Name:</strong> $fullName</p>
            <p><strong>Email:</strong> $email</p>
            <p><strong>Phone:</strong> $phone</p>
            <p><strong>Address:</strong> " . nl2br($address) . "</p>
            <p><strong>Delivery Method:</strong> " . ucfirst($deliveryMethod) . "</p>";
        
        if ($deliveryMethod === 'shipping' && !empty($shippingAddress)) {
            $adminBody .= "<p><strong>Shipping Address:</strong> " . nl2br($shippingAddress) . "</p>";
            $adminBody .= "<p><strong>Shipping Fee:</strong> \$" . number_format($shippingFee, 2) . "</p>";
        }
        
        $adminBody .= "
            <p><strong>Deposit Method:</strong> " . ucfirst(str_replace('_', ' ', $depositMethod)) . "</p>";
        
        if (!empty($message)) {
            $adminBody .= "<p><strong>Message:</strong></p><p>" . nl2br($message) . "</p>";
        }
        
        $adminBody .= "
            <p><strong>Submitted:</strong> " . date('Y-m-d H:i:s') . "</p>
            <p><strong>Applicant Location:</strong> {$geo['city']}, {$geo['region']}, {$geo['country']}</p>
            <p><strong>Applicant IP:</strong> {$geo['ip']}</p>
            <hr>
            </body></html>
        ";

        $adminSent = sendPHPMailerEmail($mail, $adminEmail, $adminSubject, $adminBody, $email, $fullName, $fullName);

        if (!$adminSent) {
            throw new Exception('Failed to send admin notification');
        }

        // Send confirmation to applicant
        $confirmSubject = "Your Puppy Reservation for $puppyName - $siteName";
        $confirmBody = "
            <html><body>
            <h2>Thank You for Your Puppy Reservation, $fullName!</h2>
            <p>We've received your reservation for $puppyName and will review it shortly.</p>
            <p><strong>Reservation Details:</strong></p>
            <ul>
                <li><strong>Puppy:</strong> $puppyName</li>
                <li><strong>Delivery:</strong> " . ucfirst($deliveryMethod) . "</li>
                <li><strong>Deposit Method:</strong> " . ucfirst(str_replace('_', ' ', $depositMethod)) . "</li>
            </ul>
            " . (!empty($message) ? "<p><strong>Message:</strong></p><p>" . nl2br($message) . "</p>" : "") . "
            <p>We'll contact you within 24 hours to discuss the next steps.</p>
            <p>If you have any questions in the meantime, feel free to reach out at $sitePhone or $email.</p>
            <hr>
            <p><small>$siteName — $siteTagline</small></p>
            <p><small>This is an automated confirmation email. Please do not reply to this message.</small></p>
            </body></html>
        ";

        sendPHPMailerEmail($mail, $email, $confirmSubject, $confirmBody);
        logSubmission('reservation', $fullName, '', $email, $phone, "Reservation for $puppyName");

        http_response_code(200);
        ob_end_clean();
        exit(json_encode(['success' => true, 'message' => 'Your puppy reservation has been received!']));
    }

    // ===== HANDLE REVIEW FORM =====
    else if ($formType === 'review') {
        $required_fields = ['name', 'email', 'message'];
        $errors = [];

        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || empty(trim($input[$field]))) {
                $errors[] = ucfirst($field) . ' is required';
            }
        }

        if (isset($input['email']) && !isValidEmail($input['email'])) {
            $errors[] = 'Invalid email address';
        }

        if (!empty($errors)) {
            http_response_code(400);
            ob_end_clean();
            exit(json_encode(['success' => false, 'message' => 'Validation failed', 'errors' => $errors]));
        }

        $name = sanitizeInput($input['name']);
        $email = sanitizeInput($input['email']);
        $rating = isset($input['rating']) ? intval($input['rating']) : 5;
        $ownerOf = isset($input['ownerOf']) ? sanitizeInput($input['ownerOf']) : '';
        $message = sanitizeInput($input['message']);

        // Get geolocation
        $geo = getClientGeolocation();

        // Build admin email
        $adminSubject = "New Review from $name";
        $adminBody = "
            <html><body>
            <h2>$siteName - New Review</h2>
            <hr>
            <p><strong>Reviewer:</strong> $name</p>
            <p><strong>Email:</strong> $email</p>
            <p><strong>Rating:</strong> $rating out of 5</p>
            " . (!empty($ownerOf) ? "<p><strong>Puppy:</strong> " . $ownerOf . "</p>" : "") . "
            <p><strong>Review:</strong></p>
            <p>" . nl2br($message) . "</p>
            <p><strong>Submitted:</strong> " . date('Y-m-d H:i:s') . "</p>
            <p><strong>Reviewer Location:</strong> {$geo['city']}, {$geo['region']}, {$geo['country']}</p>
            <p><strong>Reviewer IP:</strong> {$geo['ip']}</p>
            <hr>
            </body></html>
        ";

        $adminSent = sendPHPMailerEmail($mail, $adminEmail, $adminSubject, $adminBody, $email, $name, $name);

        if (!$adminSent) {
            throw new Exception('Failed to send admin notification');
        }

        // Optional small delay before sending confirmation to user
        if (!empty($confirmationDelayMs) && is_numeric($confirmationDelayMs) && $confirmationDelayMs > 0) {
            usleep(intval($confirmationDelayMs) * 1000);
        }

        // Send confirmation email to reviewer
        $confirmSubject = "We Received Your Review - $siteName";
        $confirmBody = "
            <html><body>
            <h2>Thanks, $name!</h2>
            <p>We received your review for $siteName. Thank you for taking the time to share your experience.</p>
            <p><strong>Your Review:</strong></p>
            <p><strong>Rating:</strong> $rating out of 5</p>
            " . (!empty($ownerOf) ? "<p><strong>Puppy:</strong> " . $ownerOf . "</p>" : "") . "
            <p>" . nl2br($message) . "</p>
            <hr>
            <p>If you have any additional photos or details to share, feel free to reply to this email and we'll follow up with you!</p>
            <p><small>$siteName — $siteTagline</small></p>
            <p><small>This is an automated confirmation email. You can reply with any photos or additional information.</small></p>
            </body></html>
        ";

        sendPHPMailerEmail($mail, $email, $confirmSubject, $confirmBody);

        // Log the submission
        logSubmission('review', $name, '', $email, '', 'Review submission');

        http_response_code(200);
        ob_end_clean();
        exit(json_encode(['success' => true, 'message' => 'Thank you — your review was sent to the site administrator.']));
    }

    // Invalid form type
    else {
        http_response_code(400);
        ob_end_clean();
        exit(json_encode(['success' => false, 'message' => 'Invalid form type']));
    }

} catch (Exception $e) {
    error_log('Form Handler Error: ' . $e->getMessage());
    http_response_code(500);
    ob_end_clean();
    exit(json_encode(['success' => false, 'message' => 'Failed to process form: ' . $e->getMessage()]));
}

/**
 * Log form submissions to a file
 */
function logSubmission($type, $firstName, $lastName, $email, $phone, $subject) {
    $logFile = __DIR__ . '/../logs/form_submissions.log';
    
    if (!is_dir(__DIR__ . '/../logs')) {
        mkdir(__DIR__ . '/../logs', 0755, true);
    }

    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'type' => $type,
        'firstName' => $firstName,
        'lastName' => $lastName,
        'email' => $email,
        'phone' => $phone,
        'subject' => $subject,
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ];

    file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND);
}
?>
